#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <math.h>

struct Imf2MIDI_CVT
{
//    struct AdLibInstrument imf_instruments[9];
//    struct AdLibInstrument imf_instrumentsPrev[9];

    /* MIDI props */
    double   midi_tempo;
    uint32_t midi_resolution;
    uint8_t  midi_mapchannel[9];
    uint32_t midi_lastpatch[9];
    uint16_t midi_lastpitch[9];
    uint32_t midi_trackBegin;
    uint32_t midi_pos;
    uint32_t midi_fileSize;
    uint32_t midi_tracksNum;
    int      midi_eventCode;
    int      midi_isEndOfTrack;
    uint32_t midi_delta;
    uint32_t midi_time;

    /* File paths */
    char    *path_in;
    char    *path_out;

    /* Flags */
    int      flag_usePitch;
    int      flag_logInstruments;
};

#define  MIDI_PITCH_CENTER      0x2000
#define  MIDI_CONTROLLER_VOLUME 7


/*****************************************************************
 *                    Bufferized output                          *
 *****************************************************************/

#define ENABLE_BUFFERIZED_WRITE
#define BUF_MAX_SIZE  20480

#ifdef ENABLE_BUFFERIZED_WRITE
static char    BUF_output[BUF_MAX_SIZE];
static size_t  BUF_stored   = 0;
static size_t  BUF_lastPos  = 0;
#endif

static void fflushb(FILE* output)
{
    #ifdef ENABLE_BUFFERIZED_WRITE
    if(BUF_stored == 0)
        return;
    fwrite(BUF_output, 1, BUF_stored, output);
    BUF_lastPos += BUF_stored;
    BUF_stored = 0;
    #else
    fflush(output);
    #endif
}

static void fseekb(FILE*f, long b)
{
    #ifdef ENABLE_BUFFERIZED_WRITE
    fflushb(f);
    #endif
    fseek(f, b, SEEK_SET);
    #ifdef ENABLE_BUFFERIZED_WRITE
    BUF_lastPos = (size_t)ftell(f);
    #endif
}

static long ftellb(FILE* file)
{
    #ifdef ENABLE_BUFFERIZED_WRITE
    (void)file;
    return (long)(BUF_lastPos + BUF_stored);
    #else
    return ftell(file);
    #endif
}

static size_t fwriteb(char* buf, size_t elements, size_t size, FILE* output)
{
    #ifdef ENABLE_BUFFERIZED_WRITE
    size_t newSize = elements * size;
    if(BUF_MAX_SIZE < (BUF_stored + newSize))
    {
        fflushb(output);
        newSize = size;
    }

    memcpy(BUF_output + BUF_stored, buf, newSize);

    BUF_stored += newSize;
    return size;
    #else
    return fwrite(buf, elements, size, output);
    #endif
}


/*****************************************************************
 *             Writing endian-specific integers                  *
 *****************************************************************/
#define write8(f, in) { uint8_t inX = (uint8_t)in; fwriteb((char*)&inX, 1, 1, (f)); }

static int writeBE16(FILE* f, uint32_t in)
{
    uint8_t bytes[2];
    bytes[1] = in & 0xFF;
    bytes[0] = (in>>8) & 0xFF;
    return (int)fwriteb((char*)bytes, 1, 2, f);
}

#if 0
static int writeLE24(FILE* f, uint32_t in)
{
    uint8_t bytes[3];
    bytes[0] = in & 0xFF;
    bytes[1] = (in>>8) & 0xFF;
    bytes[2] = (in>>16) & 0xFF;
    return (int)fwriteb((char*)bytes, 1, 3, f);
}
#endif

static int writeBE24(FILE* f, uint32_t in)
{
    uint8_t bytes[3];
    bytes[2] = in & 0xFF;
    bytes[1] = (in>>8) & 0xFF;
    bytes[0] = (in>>16) & 0xFF;
    return (int)fwriteb((char*)bytes, 1, 3, f);
}

#if 0
static int writeLE32(FILE* f, uint32_t in)
{
    uint8_t bytes[4];
    bytes[0] = in & 0xFF;
    bytes[1] = (in>>8) & 0xFF;
    bytes[2] = (in>>16) & 0xFF;
    bytes[3] = (in>>24) & 0xFF;
    return (int)fwriteb((char*)bytes, 1, 4, f);
}
#endif

static int writeBE32(FILE* f, uint32_t in)
{
    uint8_t bytes[4];
    bytes[3] = in & 0xFF;
    bytes[2] = (in>>8) & 0xFF;
    bytes[1] = (in>>16) & 0xFF;
    bytes[0] = (in>>24) & 0xFF;
    return (int)fwriteb((char*)bytes, 1, 4, f);
}

/**
 * @brief Write variable-length integer
 * @param f file to write
 * @param in input value
 * @return Count of written bytes
 */
static int writeVarLen32(FILE* f, uint32_t in)
{
    uint8_t  bytes[4];
    size_t   len = 0;
    uint8_t *ptr = bytes;
    int32_t  i;

    uint8_t isFirst = 1;
    for(i = 3; i >= 0; i--)
    {
        uint32_t b = in>>(7*i);
        uint8_t byte = (uint8_t)b & 127;
        if(!isFirst || (byte > 0) || (i == 0))
        {
            isFirst = 0;
            if(i > 0)
            {
                /* set 8th bit */
                byte |= 128;
            }
            *ptr++ = byte;
            len++;
        }
    }
    return (int)fwriteb((char*)bytes, 1, len, f);
}
/*****************************************************************/

/*****************************************************************
 *                        MIDI Writing                           *
 *****************************************************************/
static void MIDI_addDelta(struct Imf2MIDI_CVT *cvt, uint32_t delta)
{
    cvt->midi_delta += delta;
}

static void MIDI_writeHead(FILE* f, struct Imf2MIDI_CVT *cvt)
{
    fseekb(f, 0);
    fwriteb((char*)"MThd", 1, 4, f);        /* 0  */
    writeBE32(f, 6);/* Size of the head */  /* 4  */
    writeBE16(f, 0);/* MIDI format 0    */  /* 8  */
    writeBE16(f, 0);/* Zero tracks count*/  /* 10 */
    writeBE16(f, cvt->midi_resolution);     /* 12 */
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_closeHead(FILE* f, struct Imf2MIDI_CVT *cvt)
{
    fseekb(f, 10);
    writeBE16(f, cvt->midi_tracksNum);
    fflushb(f);
}


static void MIDI_writeEventCode(FILE*f,
                                struct Imf2MIDI_CVT *cvt,
                                uint8_t eventCode)
{
    if( (eventCode != cvt->midi_eventCode) || (eventCode > 0x9f) )
        write8(f, eventCode);
    cvt->midi_eventCode = eventCode;
}

static void MIDI_writeMetaEvent(FILE*f,
                                struct Imf2MIDI_CVT *cvt,
                                uint8_t type,
                                int8_t  *bytes,
                                uint32_t size)
{
    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    MIDI_writeEventCode(f, cvt, 0xFF);
    write8(f, type);
    writeVarLen32(f, size);
    fwriteb((char*)bytes, 1, (size_t)size, f);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writeControlEvent(FILE*f,
                                   struct Imf2MIDI_CVT *cvt,
                                   uint8_t channel,
                                   uint8_t controller,
                                   uint8_t value)
{
    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    channel = channel % 16;
    MIDI_writeEventCode(f, cvt, 0xB0 + channel);
    write8(f, controller);
    write8(f, value);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writePatchChangeEvent(FILE* f,
                                       struct Imf2MIDI_CVT *cvt,
                                       uint8_t channel,
                                       uint8_t patch)
{
    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    channel = channel % 16;
    MIDI_writeEventCode(f, cvt, 0xC0 + channel);
    write8(f, patch);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writePitchEvent(FILE*f,
                                 struct Imf2MIDI_CVT *cvt,
                                 uint8_t    channel,
                                 uint16_t   value)
{
    channel = channel % 9;

    if(cvt->midi_lastpitch[channel] == value)
        return;/* Don't write pitch if value is same */

    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;


    MIDI_writeEventCode(f, cvt, 0xE0 + channel);
    write8(f,  value & 0x7F);
    write8(f, (value>>7) & 0x7F);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
    /* Remember pitch value to don't repeat */
    cvt->midi_lastpitch[channel] = value;
}

static void MIDI_writeNoteOnEvent(FILE*f,
                                 struct Imf2MIDI_CVT *cvt,
                                 uint8_t    channel,
                                 uint8_t    key,
                                 uint8_t    velocity)
{
    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    channel = channel % 16;
    MIDI_writeEventCode(f, cvt, 0x90 + channel);
    write8(f,  key);
    write8(f,  velocity);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writeNoteOffEvent(FILE*f,
                                   struct Imf2MIDI_CVT *cvt,
                                   uint8_t   channel,
                                   uint8_t   key,
                                   uint8_t   velocity)
{
    uint8_t code = ((velocity != 0) ||
                   (cvt->midi_eventCode < 0) ||
                  ((cvt->midi_eventCode & 0xF0) != 0x90)) ? 0x80 : 0x90;
    channel = channel % 16;

    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    MIDI_writeEventCode(f, cvt, code + channel);
    write8(f,  key);
    write8(f,  velocity);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writeTempoEvent(FILE*f,
                                struct Imf2MIDI_CVT *cvt,
                                uint32_t ticks)
{
    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    MIDI_writeEventCode(f, cvt, 0xFF);
    write8(f, 0x51);
    write8(f, 0x03);
    writeBE24(f, ticks);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writeMetricKeyEvent(FILE*f,
                                     struct Imf2MIDI_CVT *cvt,
                                     uint8_t nom,
                                     uint8_t denom,
                                     uint8_t key1,
                                     uint8_t key2)
{
    uint8_t denomID = (uint8_t)(log((double)denom) / log(2.0));

    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    MIDI_writeEventCode(f, cvt, 0xFF);
    write8(f, 0x58);
    write8(f, 0x04);
    write8(f, nom);
    write8(f, denomID);
    write8(f, key1);
    write8(f, key2);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_writeRawEvent(FILE*f,
                               struct Imf2MIDI_CVT *cvt,
                               int msg)
{
    uint8_t byte1 = (msg & 0xFF);
    uint8_t byte2 = ((msg >> 8) & 0xFF);
    uint8_t byte3 = ((msg >> 16) & 0xFF);
    uint8_t byte4 = ((msg >> 24) & 0xFF);
    int size = 0;

    printf("byte: %02x %02x %02x %02x ", byte1, byte2, byte3, byte4);

    writeVarLen32(f, cvt->midi_delta);
    cvt->midi_delta = 0;

    switch(byte1 & 0xF0)
    {
    case 0xC0:
    case 0xD0:
        size = 1;
        printf("size 1\n");
        break;
    default:
        size = 2;
        printf("size 2\n");
        break;
    }

    write8(f,  byte1);
    if(size >= 1)
        write8(f,  (byte2) & 0xFF);
    if(size >= 2)
        write8(f,  (byte3) & 0xFF);
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}


static void MIDI_beginTrack(FILE* f, struct Imf2MIDI_CVT *cvt)
{
    if(!cvt->midi_isEndOfTrack)
        return;

    cvt->midi_time = 0;
    cvt->midi_delta = 0;
    cvt->midi_eventCode = -1;
    cvt->midi_isEndOfTrack = 0;
    fwriteb((char*)"MTrk", 1, 4, f);
    cvt->midi_trackBegin = (uint32_t)ftellb(f);
    writeBE32(f, 0); /* Track length */
    cvt->midi_tracksNum++;
    cvt->midi_fileSize = (uint32_t)ftellb(f);
}

static void MIDI_endTrack(FILE* f, struct Imf2MIDI_CVT *cvt)
{
    if(cvt->midi_isEndOfTrack)
        return;

    MIDI_writeMetaEvent(f, cvt, 0x2f, 0, 0);
    cvt->midi_isEndOfTrack = 1;
    fseekb(f, cvt->midi_trackBegin);
    writeBE32( f, cvt->midi_fileSize - cvt->midi_trackBegin - 4);
    fseekb(f, cvt->midi_fileSize);
    cvt->midi_trackBegin = 0;
}

void Imf2MIDI_init(struct Imf2MIDI_CVT *cvt)
{
    size_t i = 0;

    if(!cvt)
        return;

//    memset(cvt->imf_instruments,     0, sizeof(cvt->imf_instruments));
//    memset(cvt->imf_instrumentsPrev, 0, sizeof(cvt->imf_instrumentsPrev));
    memset(cvt->midi_mapchannel,     0, sizeof(cvt->midi_mapchannel));
    memset(cvt->midi_lastpatch,      0, sizeof(cvt->midi_lastpatch));
    memset(cvt->midi_lastpitch,      0, sizeof(cvt->midi_lastpitch));

    cvt->midi_resolution    = 384;
    cvt->midi_tempo         = 110.0;

    for(i = 0; i < 9; i++)
        cvt->midi_lastpitch[i] = MIDI_PITCH_CENTER;

    cvt->midi_trackBegin    = 0;
    cvt->midi_pos           = 0;
    cvt->midi_fileSize      = 0;
    cvt->midi_tracksNum     = 0;
    cvt->midi_eventCode     = -1;
    cvt->midi_isEndOfTrack  = 1;
    cvt->midi_delta         = 0;
    cvt->midi_time          = 0;

    cvt->path_in    = NULL;
    cvt->path_out   = NULL;

    cvt->flag_usePitch = 1;
    cvt->flag_logInstruments = 0;
}


char *songData_store = NULL;
char *songData = NULL;
uint8_t byte_411D94 = 0;

uint8_t midiVolumeValue = 127;

int songGetDelta()
{
    int v0;
    v0 = (uint8_t)*songData++;
    if ( v0 & 0x80 )
        v0 = ((uint8_t)*songData++ << 7) | v0 & 0x7F;
    return v0;
}

unsigned int runMidi(const char *outFile)
{
    char evt;
    unsigned int result;
    int msg;
    int v3;
    char v4;
    int v5;
    char v6;
    bool v7;
    struct Imf2MIDI_CVT cvt;
    FILE *file_out = fopen(outFile, "wb");

    Imf2MIDI_init(&cvt);

    cvt.midi_resolution    = 50;
    cvt.midi_tempo         = 120.0;

    MIDI_writeHead(file_out, &cvt);
    MIDI_beginTrack(file_out, &cvt);
    MIDI_writeTempoEvent(file_out, &cvt, (uint32_t)(60000000.0 / cvt.midi_tempo));
    MIDI_writeMetricKeyEvent(file_out, &cvt, 4, 4, 24, 8);

    while ( 1 )
    {
        v6 = 0;
        evt = (*songData) & 0xFF;

        if((uint8_t)evt == 0xFF)
            printf("!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!БЛИЗОК КОНЕЦ!!!!!!!!!!!!!!!!!!!!!!!!!!!!! %02x\n", evt);
        if ( (uint8_t)*songData <= 0x7Fu )
        {
            printf("меньше-равно 0x7F, заменим %02x на %02x\n", evt, byte_411D94);
            evt = byte_411D94;
        }
        else
        {
            printf("больше 0x7F, оставляем %02x\n", evt);
            ++songData;
        }
        byte_411D94 = evt;

        if ( (uint8_t)evt != 0xF0 && (uint8_t)evt != 0xF7 )// // sysEx F0 and SysEx F7
            break;

        songData += songGetDelta();                 // // Variable-length value byte offset
LABEL_22:

        if ( !v6 )
        {
            v5 = songGetDelta();
            if ( v5 )
            {
                printf("tick: %d\n", v5);
                MIDI_addDelta(&cvt, v5);
//                lo_qword_411D7C = 10 * v5 + qword_411D7C;
//                result = timeGetTime();
//                if(result < (unsigned int)qword_411D7C )
//                    return result;
            }
        }
    }

    if ( (uint8_t)evt != 255 )
    {
        msg = (uint8_t)evt | ((uint8_t)*songData++ << 8);
        v7 = 1;
        v3 = evt & 0xF0;

        if ( v3 == 128 || v3 == 144 )
        {
            v7 = (uint8_t)midiVolumeValue >= 1u;
        }
        else if ( v3 == 192 || v3 == 208 )
        {
LABEL_20:
            if(v7)
            {
                printf("-> %x\n", msg);
                MIDI_writeRawEvent(file_out, &cvt, msg);
                //midiOutShortMsg(hmo, msg);
            }
            goto LABEL_22;
        }

        v4 = *songData++;
        msg = (uint16_t)msg | ((v4 & 0x7F) << 16);
        v6 = v4 & 0x80;
        goto LABEL_20;
    }

//    if(unkSongId == 0xFFFF)
//    {
//        result = midiReset();
//    }
//    else
//    {
//        unkCurSongId = unkSongId;
//        result = (unsigned int)songDataBegin;
//        songData = songDataBegin;
//    }

    MIDI_endTrack(file_out, &cvt);
    MIDI_closeHead(file_out, &cvt);

    fclose(file_out);

    return result;
}

void convert(const char *in_file, const char *out_file)
{
    FILE *in;
    in = fopen(in_file, "rb");
    if(!in)
        return;

    songData_store = (char*)malloc(100000);
    fread(songData_store, 1, 100000, in);
    fclose(in);
    byte_411D94 = 0;
    midiVolumeValue = 127;
    songData = songData_store;
    runMidi(out_file);
    free(songData_store);
}

int main()
{
    char in_name[256];
    char out_name[256];
    int i;

    for(i = 0; i < 50; i++)
    {
        sprintf(in_name, "level%d.bin", i);
        sprintf(out_name, "SolGame-%d.mid", i);
        convert(in_name, out_name);

        sprintf(in_name, "level%d.1.bin", i);
        sprintf(out_name, "SolGame-%d.1.mid", i);
        convert(in_name, out_name);
    }
    printf("done\n");

    return 0;
}
